sampler2D color_tex					: register(s0);	///< color tex
sampler2D normal_tex				: register(s1);	///< normal tex
sampler2D depth_tex					: register(s4);	///< depth tex
sampler2D small_tex					: register(s9);	///< small tex
sampler2D small_tex_pc0				: register(s0);	///< small tex (for PC, bound to sampler 0)
sampler2D small_tex_pc1				: register(s1);	///< small tex (for PC, bound to sampler 1)
sampler2D small_tex_pc2				: register(s2);	///< small tex (for PC, bound to sampler 2)
sampler2D small_tex_pc3				: register(s3);	///< small tex (for PC, bound to sampler 3)

#include "VertexShaderSemantics.fx"
#define g_ViewportScaleBias		vs_viewportScaleBias		// viewport scale xy, offset xy

#ifndef SM1

uniform float2 motionFactors		: register(c128);	///< contains {fadeLimit, fadeScale}
uniform float3 motionVector			: register(c129);	///< contains clip space camera motion vector in xyz

#else

uniform float2 motionFactors		: register(c0);	///< contains {fadeLimit, fadeScale}
uniform float4 motionVector			: register(c1);	///< contains {interest_x, interest_y, blurFactor_x, blurFactor_y}

#endif


struct VS_IN
{
	float4 position : POSITION;
#ifdef _WINPC
	float2 tex		: TEXCOORD0;
#endif
};

struct VS_OUT
{
	float4 position		: POSITION;
	half3 Texture		: TEXCOORD0;
	half4 offset_0_1	: TEXCOORD1;
	half4 offset_2_3	: TEXCOORD2;
	half4 offset_4_5	: TEXCOORD3;
};

struct PS_IN
{
	half3 Texture		: TEXCOORD0;
	half4 offset_0_1	: TEXCOORD1;
	half4 offset_2_3	: TEXCOORD2;
	half4 offset_4_5	: TEXCOORD3;
};

//_________________________________________________________________________________________________


#ifdef CG_PS3

inline half readDepthTexture(sampler2D tex, half2 uvTex)
{
	half4 depthColor = tex2D(tex, uvTex).rgba;
	float4 depthFactor = float4(256.0/16777215.0, 1.0/16777215.0, 0.0f, 256.0*256.0/16777215.0);
	return dot( round(depthColor*255.0), depthFactor );
}

#else

inline half readDepthTexture(sampler2D tex, half2 uvTex)
{
	return tex2D(tex, uvTex).r;
}

#endif

//_________________________________________________________________________________________________

inline half4 speedblur(PS_IN In, half4 srcColor)
{
	// sample 6 times the lowres color tex to build the motion blur
	half4 blended;
	blended  = tex2D(small_tex, In.offset_0_1.xy);
	blended += tex2D(small_tex, In.offset_0_1.zw);
	blended += tex2D(small_tex, In.offset_2_3.xy);
	blended += tex2D(small_tex, In.offset_2_3.zw);
	blended += tex2D(small_tex, In.offset_4_5.xy);
	blended += tex2D(small_tex, In.offset_4_5.zw);
	
	blended /= 6.0;
	
	half blendFactor = In.Texture.z;
	return lerp(srcColor, blended, blendFactor);
}

//_________________________________________________________________________________________________

/**
 * Entry point for the vertex shader.
 */
VS_OUT speedblur_vx(VS_IN input)
{
	VS_OUT output;
	output.position = input.position;
#ifdef _WINPC
	output.Texture.xy = input.tex;
#else
	output.Texture.xy = input.position.xy * half2(0.5, -0.5) + 0.5;
#endif
	
	output.Texture.xy = output.Texture.xy * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	half2 convergencePoint = 0.5;
	half2 pullVector = output.Texture.xy - convergencePoint;
	half2 panVector = -motionVector.xy;

	// this scales down and set the pull vector to 0 within a ellipsis in the middle
	// of the screen, as ramping up the pull vector on the edge of the screen
	pullVector *= smoothstep(0.03, 0.5, length(pullVector));

	half2 delta = panVector + pullVector * motionVector.z;

	output.offset_0_1.xy = output.Texture.xy + delta;
	output.offset_0_1.zw = output.Texture.xy + delta * 2.0;
	output.offset_2_3.xy = output.Texture.xy + delta * 3.0;
	output.offset_2_3.zw = output.Texture.xy + delta * 4.0;
	output.offset_4_5.xy = output.Texture.xy + delta * 5.0;
	output.offset_4_5.zw = output.Texture.xy + delta * 6.0;

	// blend factor
	half fadeScale = motionFactors.y;
	output.Texture.z = saturate(dot(delta, delta) * fadeScale);

	return output;
}

//_________________________________________________________________________________________________

/**
 * Entry point for the pixel shader.
 */
half4 speedblur_px( PS_IN In ) : COLOR
{
	return speedblur(In, tex2D(color_tex, In.Texture.xy));
}

//_________________________________________________________________________________________________

/**
 * Entry point for the pixel shader, with depth restore.
 */
void speedblurdepth_px( PS_IN In, out float4 outColor : COLOR, out float outDepth : DEPTH)
{
	outColor = speedblur(In, tex2D(color_tex, In.Texture.xy));

	// restore depth texture
	outDepth = readDepthTexture(depth_tex, In.Texture.xy);
}

//_________________________________________________________________________________________________

/**
 * Entry point for the pixel shader, with normal and depth buffers restore.
 */
void speedblurnormaldepth_px(	PS_IN In,
								out float4 outColor : COLOR0,
								out float4 outNormal : COLOR1,
								out float outDepth : DEPTH)
{
	outColor = speedblur(In, tex2D(color_tex, In.Texture.xy));
	outNormal = tex2D(normal_tex, In.Texture.xy);
	outDepth = readDepthTexture(depth_tex, In.Texture.xy);
}

//_________________________________________________________________________________________________

//_________________________________________________________________________________________________

/**
 * Entry point for the PC pixel shader.
 */
half4 speedblur_pc_px( PS_IN In ) : COLOR
{
	// sample 6 times the lowres color tex to build the motion blur
	half4 blended;
	blended  = tex2D(small_tex_pc0, In.offset_0_1.xy);
	blended += tex2D(small_tex_pc0, In.offset_0_1.zw);
	blended += tex2D(small_tex_pc0, In.offset_2_3.xy);
	blended += tex2D(small_tex_pc0, In.offset_2_3.zw);
	blended += tex2D(small_tex_pc0, In.offset_4_5.xy);
	blended += tex2D(small_tex_pc0, In.offset_4_5.zw);
	
	blended /= 6.0;

	return float4(blended.rgb, In.Texture.z);
	
}

//_________________________________________________________________________________________________

#ifdef _WINPC

// SM1 shaders

struct VS_OUT_SM1
{
	float4 position		: POSITION;
	float4 color		: COLOR;
	float2 texcoord0	: TEXCOORD0;
	float2 texcoord1	: TEXCOORD1;
	float2 texcoord2	: TEXCOORD2;
	float2 texcoord3	: TEXCOORD3;
};

VS_OUT_SM1 speedblur_SM1_vx(VS_IN input)
{
	VS_OUT_SM1 output;
	output.position = input.position;

	float2 baseTexCoord = input.tex;

	baseTexCoord = baseTexCoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	half2 convergencePoint = 0.5;
	half2 pullVector = baseTexCoord.xy - convergencePoint;
	half2 panVector = -motionVector.xy;

	// this scales down and set the pull vector to 0 within a ellipsis in the middle
	// of the screen, as ramping up the pull vector on the edge of the screen
	pullVector *= smoothstep(0.1, 0.5, length(pullVector));

	half2 delta = panVector + pullVector * motionVector.z;

	output.texcoord0 = baseTexCoord + delta * 1.0;
	output.texcoord1 = baseTexCoord + delta * 2.0;
	output.texcoord2 = baseTexCoord + delta * 3.0;
	output.texcoord3 = baseTexCoord + delta * 4.0;

	output.color = saturate(dot(delta, delta) * motionFactors.y);

	return output;
}

float4 speedblur_SM1_px( VS_OUT_SM1 IN ) : COLOR
{
	float4 blended = 0;
	blended += tex2D(small_tex_pc0, IN.texcoord0) * 0.25;
	blended += tex2D(small_tex_pc1, IN.texcoord1) * 0.25;
	blended += tex2D(small_tex_pc2, IN.texcoord2) * 0.25;
	blended += tex2D(small_tex_pc3, IN.texcoord3) * 0.25;
	
	return float4(blended.rgb, IN.color.a);
}

#endif // _WINPC
